#include "config.h"

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif
#include <sys/types.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include <sys/socket.h>
#include <netinet/in.h>
#ifdef HAVE_ARPA_INET_H
# include <arpa/inet.h>
#endif
#include <stdio.h>
#include <string.h>
#include <errno.h>
#if STDC_HEADERS
# define bzero(b,n) memset(b,0,n)
#else
# include <strings.h>
# ifndef HAVE_STRCHR
#  define strchr index
# endif
# ifndef HAVE_MEMCPY
#  define memcpy(d, s, n) bcopy ((s), (d), (n))
#  define memmove(d, s, n) bcopy ((s), (d), (n))
# endif
#endif
#ifdef HAVE_CTYPE_H
# include <ctype.h>
#endif
#ifndef HAVE_INET_ATON
extern int inet_aton (const char *, struct in_addr *);
#endif

#include "rawsend.h"

#define PDU_SIZE 5000

#define FLOWPORT 2000 

#define DEFAULT_SOCKBUFLEN 65536

#define PORT_SEPARATOR	'/'
#define FREQ_SEPARATOR	'/'
#define TTL_SEPARATOR	','

#define MAX_PEERS 100
#define MAX_LINELEN 1000

#include <netinet/sctp.h>


enum peer_flags
{
  pf_SPOOF	= 0x0001,
  pf_CHECKSUM	= 0x0002,
};

struct peer {
  int			fd;
  struct sockaddr_in	addr;
  int			port;
  int			freq;
  int			freqcount;
  int			ttl;
  enum peer_flags	flags;
};

struct samplicator_context {
  struct source_context *sources;
  struct in_addr	faddr;
  int			fport;
  long			sockbuflen;
  int			debug;
  int			fork;
  enum peer_flags	defaultflags;

  int			fsockfd;
  int			fsockfd_sctp;
  int			proto; // 0 = UDP; 1 = SCTP; 2 = mixed;
};


struct source_context {
  struct source_context *next;
  struct in_addr	source;
  struct in_addr	mask;
  struct peer	      *	peers;
  unsigned		npeers;
  unsigned		tx_delay;
  int			debug;
};

static void usage(const char *);
static int send_pdu_to_peer (struct peer *, const void *, size_t,
			     struct sockaddr_in *);
static void parse_args (int, char **, struct samplicator_context *, struct source_context *);
static int parse_peers (int, char **, struct samplicator_context *, struct source_context *);
static int init_samplicator (struct samplicator_context *);
static int samplicate (struct samplicator_context *);
static int make_cooked_udp_socket (long);
static void read_cf_file (const char *, struct samplicator_context *);

/* Work around a GCC compatibility problem with respect to the
   inet_ntoa() system function */
#undef inet_ntoa
#define inet_ntoa(x) my_inet_ntoa(&(x))

static const char *
my_inet_ntoa (const struct in_addr *in)
{
  unsigned a = ntohl (in->s_addr);
  static char buffer[16];
  sprintf (buffer, "%d.%d.%d.%d",
	   (a >> 24) & 0xff,
	   (a >> 16) & 0xff,
	   (a >> 8) & 0xff,
	   a & 0xff);
  return buffer;
}

int main(argc, argv)
int argc;
char **argv;
{
  struct samplicator_context ctx;
  struct source_context cmd_line;

  cmd_line.source.s_addr = 0;
  cmd_line.mask.s_addr = 0;
  cmd_line.npeers = 0;
  ctx.sources = &cmd_line;

  cmd_line.next = (struct source_context *) NULL;

  parse_args (argc, argv, &ctx, &cmd_line);

  if (init_samplicator (&ctx) == -1)
    exit (1);
  if (samplicate (&ctx) != 0) /* actually, samplicate() should never return. */
    exit (1);
  exit (0);
}

static void
parse_args (argc, argv, ctx, sctx)
     int argc;
     char **argv;
     struct samplicator_context *ctx;
     struct source_context *sctx;
{
  extern char *optarg;
  extern int errno, optind;
  int i;

  ctx->sockbuflen = DEFAULT_SOCKBUFLEN;
  ctx->faddr.s_addr = htonl (INADDR_ANY);
  ctx->fport = FLOWPORT;
  ctx->debug = 0;
  ctx->fork = 0;
  ctx->sources = NULL;
  ctx->defaultflags = pf_CHECKSUM;
  ctx->proto = 0; // default => UDP
  /* assume that command-line supplied peers want to get all data */
  sctx->source.s_addr = 0;
  sctx->mask.s_addr = 0;

  sctx->tx_delay = 0;

  while ((i = getopt (argc, argv, "hb:d:p:s:x:c:fSnt")) != -1)
    {
      switch (i)
	{
	case 'b': /* buflen */
	  ctx->sockbuflen = atol (optarg);
	  break;
	case 'd': /* debug */
	  ctx->debug = atoi (optarg);
	  break;
	case 'n': /* no UDP checksums */
	  ctx->defaultflags &= ~pf_CHECKSUM;
	  break;
	case 'p': /* flow port */
	  ctx->fport = atoi (optarg);
	  if (ctx->fport < 0
	      || ctx->fport > 65535)
	    {
	      fprintf (stderr,
		       "Illegal receive port %d - \
should be between 0 and 65535\n",
		       ctx->fport);
	      usage (argv[0]);
	      exit (1);
	    }
	  break;
	case 's': /* flow address */
	  if (inet_aton (optarg, &ctx->faddr) == 0)
	    {
	      fprintf (stderr, "parsing IP address (%s) failed\n", optarg);
	      usage (argv[0]);
	      exit (1);
	    }
	  break;
	case 'x': /* transmit delay */
	  sctx->tx_delay = atoi (optarg);
	  break;
	case 'S': /* spoof */
	  ctx->defaultflags |= pf_SPOOF;
	  break;
	case 'c': /* config file */
	  read_cf_file(optarg,ctx);
	  break;
	case 'f': /* fork */
	  ctx->fork = 1;
	  break;
	case 't': /* use SCTP instead of UDP */
	  fprintf(stderr, "Using SCTP instead of UDP\n");
	  ctx->proto = 1;
	  break;
	case 'h': /* help */
	  usage (argv[0]);
	  exit (0);
	  break;
	default:
	  usage (argv[0]);
	  exit (1);
	  break;
	}
    }

  if (argc - optind > 0)
    {
      if (parse_peers (argc - optind, argv + optind, ctx, sctx) == -1)
	{
	  usage (argv[0]);
	  exit (1);
	}
    }
}

static int
parse_peers (argc, argv, ctx, sctx)
     int argc;
     char **argv;
     struct samplicator_context *ctx;
     struct source_context *sctx;
{
  int i;
  char tmp_buf[256];
  static int cooked_sock = -1;
  static int raw_sock = -1;
  char *c;
  struct source_context *ptr;

  /* allocate for argc peer entries */
  sctx->npeers = argc;

  if (!(sctx->peers = (struct peer*) malloc (sctx->npeers * sizeof (struct peer)))) {
    fprintf(stderr, "malloc(): failed.\n");
    return -1;
  }

  /* zero out malloc'd memory */
  bzero(sctx->peers, sctx->npeers*sizeof (struct peer));

  /* fill in peer entries */
  for (i = 0; i < argc; ++i)
    {
      sctx->peers[i].flags = ctx->defaultflags;
	
      if (strlen (argv[i]) > 255)
	{
	  fprintf (stderr, "ouch!\n");
	  return -1;
	}
      strcpy (tmp_buf, argv[i]);

      /* skip to end or port seperator */
      for (c = tmp_buf; (*c != PORT_SEPARATOR) && (*c); ++c);

      /* extract the port part */
      if (*c == PORT_SEPARATOR)
	{
	  int port;
	  *c = 0;
	  ++c;
	  port = atoi(c);
	  if (port < 0 || port > 65535)
	    {
	      fprintf (stderr, "Illegal destination port %d - \
should be between 0 and 65535\n", port);
	      return -1;
	    }
	  sctx->peers[i].addr.sin_port = htons (port);
	}
      else 
	sctx->peers[i].addr.sin_port = htons (FLOWPORT);

      /* extract the frequency part */
      sctx->peers[i].freqcount = 0;
      sctx->peers[i].freq = 1;
      for (; (*c != FREQ_SEPARATOR) && (*c); ++c)
	if (*c == TTL_SEPARATOR) goto TTL;
      if (*c == FREQ_SEPARATOR)
	{
	  *c = 0;
	  ++c;
	  sctx->peers[i].freq = atoi(c);
	}

      /* printf("Frequency: %d\n", sctx->peers[i].freq); */

      /* extract the TTL part */
      for (; (*c != TTL_SEPARATOR) && (*c); ++c); 
    TTL:   
      if ((*c == TTL_SEPARATOR) && (*(c+1) > 0))
        {
          *c = 0;
          ++c;
          sctx->peers[i].ttl = atoi (c);
	  if (sctx->peers[i].ttl < 1
	      || sctx->peers[i].ttl > 255)
	    {
	      fprintf (stderr,
		       "Illegal value %d for TTL - should be between 1 and 255.\n",
		       sctx->peers[i].ttl);
	      return -1;
	    }
        }
      else
        sctx->peers[i].ttl = DEFAULT_TTL; 

      /* extract the ip address part */
      if (inet_aton (tmp_buf, & sctx->peers[i].addr.sin_addr) == 0)
	{
	  fprintf (stderr, "parsing IP address (%s) failed\n", tmp_buf);
	  return -1;
	}

      sctx->peers[i].addr.sin_family = AF_INET;

      if (sctx->peers[i].flags & pf_SPOOF)
	{
	  if (raw_sock == -1)
	    {
	      if ((raw_sock = make_raw_udp_socket (ctx->sockbuflen)) < 0)
		{
		  if (errno == EPERM)
		    {
		      fprintf (stderr, "Not enough privilege for -S option---try again as root.\n");
		    }
		  else
		    {
		      fprintf (stderr, "creating raw socket: %s\n", strerror(errno));
		    }
		  return -1;
		} 
	    }
	  sctx->peers[i].fd = raw_sock;
	}
      else
	{
	  if (cooked_sock == -1)
	    {
	      if ((cooked_sock = make_cooked_udp_socket (ctx->sockbuflen)) < 0)
		{
		  fprintf (stderr, "creating cooked socket: %s\n",
			   strerror(errno));
		  return -1;
		}
	    }
	  sctx->peers[i].fd = cooked_sock;
	}
    }
  if (ctx->sources == NULL)
    {
      ctx->sources = sctx;
    }
  else
    {
      for (ptr = ctx->sources; ptr->next != NULL; ptr = ptr->next);
      ptr->next = sctx;
    } 
  return 0;
}

// Trim whitespaces from the left of a string
unsigned char * trim(unsigned char *c) {
    unsigned char *tmp;

    tmp = c;
    while ( tmp[0] == ' ' || tmp[0] == '\t' || tmp[0] == '\n' || tmp[0] == '\r' ) {
	printf("KO = %u\n", tmp[0]);
	tmp += 1;
    }

    return tmp;
}

static void
read_cf_file (file, ctx)
     const char *file;
     struct samplicator_context *ctx;
{
  FILE *cf;
  char tmp_s[MAX_LINELEN];
  int argc;
  char *argv[MAX_PEERS];
  unsigned char *c, *slash, *e, *tmpc;
  struct source_context *sctx;

  ctx->proto = 0;

  if ((cf = fopen(file,"r")) == NULL)
    {
      fprintf(stderr, "read_cf_file: cannot open %s. Aborting.\n",file);
      exit(1);
    }

  while (!feof(cf))
    {
      fgets(tmp_s, MAX_LINELEN - 1, cf);

      if ((c = strchr(tmp_s, '#')) != 0)
	continue;
	
      /* lines look like this:

      [sctp] ipadd[/mask]: dest[:port[/freq][,ttl]]  dest2[:port2[/freq2][,ttl2]]...

      */

      if ( (tmpc = (unsigned char *)strcasestr(tmp_s, "sctp")) != NULL ) {
	if ( ctx->proto == 0 ) {
	    ctx->proto = 1;
	}
	tmpc += sizeof("sctp");
      } else {
        tmpc = tmp_s;
	if ( ctx->proto > 0 ) {
	    ctx->proto = 2; // mixed UDP & SCTP protocol mode
	}
      }

      tmpc = trim(tmpc);
	
      if ((c = strchr(tmpc, ':')) != 0)
	{
	  *c++ = 0;

	  sctx = calloc(1, sizeof(struct source_context));
	  if ((slash = strchr (tmpc, '/')) != 0)
	    {
	      *slash++ = 0;
		
	      /* fprintf (stderr, "parsing IP address mask (%s)\n",slash); */
	      if (inet_aton (slash, &(sctx->mask)) == 0)
		{
             	  fprintf (stderr, "parsing IP address mask (%s) failed\n",slash);
        	  exit (1);
		}
	    }
	  else
	    {
	      inet_aton ("255.255.255.255", &(sctx->mask));
	    }
	  /* fprintf (stderr, "parsing IP address (%s)\n",tmpc); */
  	  if (inet_aton (tmpc, &(sctx->source)) == 0)
            {
	      fprintf (stderr, "parsing IP address (%s) failed\n",tmpc);
	      exit (1);
	    }

	  /*	
	    fprintf (stderr, "parsed into %s/",
	    inet_ntoa(sctx->source));
	    fprintf (stderr, "%s\n",
	    inet_ntoa(sctx->mask));
	  */

	  argc = 0;
	  while (*c != 0)
	    {
	      while ((*c != 0) && isspace ((int) *c))
		c++;
	      if (*c == 0 ) break;

	      e = c;
	      while((*e != 0) && !isspace ((int) *e))
		e++;
	      argv[argc++] = c;
	      c = e;
	      if (*c != 0)
		c++;
	      *e = 0;
	    }
	  if (argc > 0) 
	    {
	      if (parse_peers (argc, argv, ctx, sctx) == -1)
		{
		  usage (argv[0]);
		  exit (1);
		}
	    }
	}
    }
  fclose(cf);


  if ( ctx->debug ) {
    fprintf(stderr, "PROTOCOL MODE: ");
    switch(ctx->proto) {
	case 0:
	    fprintf(stderr, "UDP\n");
	    break;
	case 1:
	    fprintf(stderr, "SCTP\n");
	    break;
	case 2:
	    fprintf(stderr, "UDP and SCTP mixed\n");
	    break;
	default:
	    break;
    }
  }
//printf("ctx->proto = %d\n", ctx->proto);

}

/* init_samplicator: prepares receiving socket */
static int
init_samplicator (ctx)
     struct samplicator_context *ctx;
{
  struct sockaddr_in local_address, local_address2;

  ctx->fsockfd = 0;
  ctx->fsockfd_sctp = 0;

  /* setup to receive flows */
  bzero (&local_address, sizeof local_address);
  local_address.sin_family = AF_INET;
  local_address.sin_addr.s_addr = ctx->faddr.s_addr;
//printf("fport = %d\n", ctx->fport);
  local_address.sin_port = htons (ctx->fport);

 // Initiate UDP receive socket if needed
 if ( ctx->proto == 0 || ctx->proto == 2 ) {

  if ((ctx->fsockfd = socket (AF_INET, SOCK_DGRAM, 0)) < 0)
    {
      fprintf (stderr, "socket(): %s\n", strerror (errno));
      return -1;
    } 
  if (setsockopt (ctx->fsockfd, SOL_SOCKET, SO_RCVBUF,
		  (char *) &ctx->sockbuflen, sizeof ctx->sockbuflen) == -1)
    {
      fprintf (stderr, "setsockopt(SO_RCVBUF,%ld): %s\n",
	       ctx->sockbuflen, strerror (errno));
    }

  if (bind (ctx->fsockfd,
	    (struct sockaddr*)&local_address, sizeof local_address) < 0)
    {
      fprintf (stderr, "bind(): %s\n", strerror (errno));
      return -1;
    }
    if (ctx->debug) {
	fprintf (stderr, "SOCKET: UDP socket initialized\n");
    } 
  }

 // Initiate SCTP receive socket if needed
 if ( ctx->proto > 0 ) {

    if ((ctx->fsockfd_sctp = socket(AF_INET, SOCK_SEQPACKET, IPPROTO_SCTP)) < 0) {
	fprintf(stderr, "socket(): %s\n", strerror(errno));
	return -1;
    }

    if (bind(ctx->fsockfd_sctp, (struct sockaddr*)&local_address, sizeof(local_address)) < 0) {
	fprintf(stderr, "bind(): %s\n", strerror(errno));
	return -1;
    }

    listen(ctx->fsockfd_sctp, 1);
    if (ctx->debug) {
	fprintf(stderr, "SOCKET: Listening on SCTP socket\n");
    }
 }

  return 0;
}

static int udp_send(struct samplicator_context *ctx, struct source_context *sctx, unsigned char *fpdu, struct sockaddr_in *remote_address, unsigned int nread, int len) {
  unsigned int n, i;

  if (nread > PDU_SIZE)
    {
      fprintf (stderr, "Warning: %d excess bytes discarded\n", n-PDU_SIZE);
	n = PDU_SIZE;
    } else {
	n = nread;
    }
  if (len != sizeof(struct sockaddr_in))
    {
      fprintf (stderr, "recvfrom() return address length %d - expected %d\n",
	   len, sizeof(struct sockaddr_in));
      exit (1);
    }
  if (ctx->debug)
    {
      fprintf (stderr, "Received %d bytes from %s:%d\n", n,
      inet_ntoa (remote_address->sin_addr), (int) ntohs (remote_address->sin_port));
    }

  for (sctx = ctx->sources; sctx != NULL; sctx = sctx->next)
    {
      if ((sctx->source.s_addr == 0)
          || ((remote_address->sin_addr.s_addr & sctx->mask.s_addr)
		  == sctx->source.s_addr))
      for (i = 0; i < sctx->npeers; ++i)
          {
	    if (sctx->peers[i].freqcount == 0)
	      {
	        if (send_pdu_to_peer(& (sctx->peers[i]), (void *)fpdu, n, remote_address)
			== -1)
		  {
		    fprintf (stderr, "Sending datagram to %s:%d failed: %s\n",
				 inet_ntoa (sctx->peers[i].addr.sin_addr),
				 (int) ntohs (sctx->peers[i].addr.sin_port),
				 strerror (errno));
		  }
		else if (ctx->debug)
		  {
		    fprintf (stderr, "+Sent to %s:%d\n",
				 inet_ntoa (sctx->peers[i].addr.sin_addr),
				 (int) ntohs (sctx->peers[i].addr.sin_port)); 
	          }
		    sctx->peers[i].freqcount = sctx->peers[i].freq-1;
	      }
	      else
	      {
	        --sctx->peers[i].freqcount;
	      }
	      if (sctx->tx_delay)
	      usleep (sctx->tx_delay);
      }
      else
      {
          if (ctx->debug)
	    {
	      fprintf (stderr, "-Not matching %s/", inet_ntoa(sctx->source));
	      fprintf (stderr, "%s\n", inet_ntoa(sctx->mask));
	    }
      }
  }

  return 0;
}

static int
samplicate (ctx)
     struct samplicator_context *ctx;
{
  unsigned char fpdu[PDU_SIZE];
  unsigned char fpdu_sctp[PDU_SIZE];
  struct sockaddr_in remote_address;
  struct source_context *sctx;
  pid_t pid;
  int i, len, n;
  struct sctp_event_subscribe sctp_events;
  unsigned int ulen;
  int flags;
  unsigned int true = 1;
  fd_set rset;

  int sctp_control = 0;
  int highsock = 0;

// SCTP msghdr
  struct msghdr msg_header;
  char cdata[sizeof(struct sctp_sndrcvinfo) + sizeof(struct cmsghdr) + 30];
  char data[8192];
  struct sockaddr_in from;
  socklen_t fromlen = sizeof(struct sockaddr_in);
  struct iovec iov;
  struct cmsghdr *cmsg;

  // SCTP Notification
  struct sctp_sndrcvinfo *sinfo;
  union sctp_notification *snp;
  struct sctp_assoc_change *sac;
  int idx = 0;

  if ( ctx->proto > 0 ) {
    len = sizeof(remote_address);

    /* Subscribe for SCTP socket events if debug mode is turned on*/
    memset((void *)&sctp_events, 0, sizeof(sctp_events));
    if ( ctx->debug ) {
	sctp_events.sctp_data_io_event = 1;
        sctp_events.sctp_association_event = 1;
	sctp_events.sctp_address_event = 1;
	sctp_events.sctp_send_failure_event = 1;
	sctp_events.sctp_peer_error_event = 1;
	sctp_events.sctp_shutdown_event = 1;
	sctp_events.sctp_partial_delivery_event = 1;
	sctp_events.sctp_adaption_layer_event = 1;
	if ( setsockopt(ctx->fsockfd_sctp, IPPROTO_SCTP, SCTP_EVENTS, (const void *)&sctp_events, sizeof(sctp_events)) < 0 ) {
	    fprintf(stderr, "setsockopt() SCTP_EVENTS failed\n");
	}
    }
    if ( setsockopt(ctx->fsockfd_sctp, IPPROTO_SCTP, SCTP_NODELAY, &true, sizeof(true)) < 0 ) {
	fprintf(stderr, "setsockopt() SCTP_NODELAY failed\n");
    }
    printf("\n");
  }

  /* check is there actually at least one configured data receiver */
  for (i = 0, sctx = ctx->sources; sctx != NULL; sctx = sctx->next)
    if (sctx->npeers > 0)  i += sctx->npeers; 
  if (i == 0)
    {
      fprintf(stderr, "You have to specify at least one receiver, exiting\n");
      exit(1);
    }

  if (ctx->fork == 1)
    {
      pid = fork();
      if (pid == -1)
        {
          fprintf (stderr, "failed to fork process\n");
          exit (1);
        }
      else if (pid > 0)
        { /* kill the parent */
          exit (0);
        }
      else
        { /* end interaction with shell */
          fclose(stdin);
          fclose(stdout);
          fclose(stderr);
        }
    }


  if ( ctx->fsockfd > ctx->fsockfd_sctp ) {
    highsock = ctx->fsockfd;
  } else {
    highsock = ctx->fsockfd_sctp;
  }
  FD_ZERO(&rset);
  while (1)
    {
      len = sizeof(remote_address);

      if ( ctx->proto == 0 ) {
	FD_SET(ctx->fsockfd, &rset);
      }
      if ( ctx->proto == 1 ) {
	FD_SET(ctx->fsockfd_sctp, &rset);
      }
      if ( ctx->proto == 2 ) {
	FD_SET(ctx->fsockfd, &rset);
	FD_SET(ctx->fsockfd_sctp, &rset);
      }

      if ( select(highsock+1, &rset, NULL, NULL, NULL) < 0 ) {
	fprintf(stderr, "select(): failed\n");
	exit(1);
      }

      // UDP socket handling if needed
	if ( FD_ISSET(ctx->fsockfd, &rset) ) {
	    if ((n = recvfrom (ctx->fsockfd, (char*)fpdu, sizeof (fpdu), 0, (struct sockaddr*) &remote_address, &len)) == -1) {
		fprintf(stderr, "recvfrom(): %s\n", strerror(errno));
		exit (1);
	    }
	    if ( ctx->debug ) {
		fprintf(stderr, "*** UDP message received ***\n");
	    }
	    udp_send(ctx, sctx, (unsigned char*)fpdu, &remote_address, n, len);
	}

	if ( FD_ISSET(ctx->fsockfd_sctp, &rset) ) {

	    sctp_control = 0;
	    bzero(cdata, sizeof(cdata));
	    bzero(fpdu_sctp, sizeof(fpdu_sctp));
	    bzero(&msg_header, sizeof(msg_header));

	    msg_header.msg_name = &remote_address;
	    msg_header.msg_namelen = sizeof(remote_address);
	    msg_header.msg_iov = &iov;
	    msg_header.msg_iovlen = 1;
	    msg_header.msg_control = cdata;
	    msg_header.msg_controllen = sizeof(cdata);
	    iov.iov_base = fpdu_sctp;
	    iov.iov_len = sizeof(fpdu_sctp);

	    cmsg = (struct cmsghdr *)cdata;
	    sinfo = (struct sctp_sndrcvinfo *)(cmsg + 1);

	    if ( (n = recvmsg(ctx->fsockfd_sctp, &msg_header, MSG_DONTWAIT)) == -1 ) {
		fprintf(stderr, "recvmsg(): %s\n", strerror(errno));
		exit (1);
    	    }

	    if ( ctx->debug ) {

		fprintf(stderr, "*** SCTP message received ***\n");
		fprintf(stderr, "  Source address: %s:%d\n", inet_ntoa(remote_address.sin_addr), ntohs(remote_address.sin_port));
		fprintf(stderr, "  Message size: %u\n", n);

		fprintf(stderr, "  Stream: %d\n", sinfo->sinfo_stream);
		fprintf(stderr, "  Stream Seq Number: %d\n", sinfo->sinfo_ssn);
		fprintf(stderr, "  Association ID: %d\n", sinfo->sinfo_assoc_id);
		fprintf(stderr, "  SCTP flags: 0x%X\n", sinfo->sinfo_flags);
		fprintf(stderr, "  Payload Protocol ID: %d\n", sinfo->sinfo_ppid);
		fprintf(stderr, "  Transmission Seq Number: 0x%X\n", sinfo->sinfo_tsn);
		fprintf(stderr, "  Cumulative TSN: %u\n", sinfo->sinfo_cumtsn);
	    }


	    if ( MSG_NOTIFICATION & msg_header.msg_flags ) {

		sctp_control = 1;
		snp = (union sctp_notification *)msg_header.msg_iov[idx].iov_base;

		switch ( snp->sn_header.sn_type ) {
		    case SCTP_ASSOC_CHANGE:
			sac = &snp->sn_assoc_change;
			if ( ctx->debug ) {
			    fprintf(stderr, "  * Notification: SCTP_ASSOC_CHANGE (state=%u, inbound streams=%u, outbound streams=%u, error=%u)\n", sac->sac_state, sac->sac_inbound_streams, sac->sac_outbound_streams, sac->sac_error);
			}
			break;
		    case SCTP_PEER_ADDR_CHANGE:
			if ( ctx->debug ) {
			    fprintf(stderr, "  * Notification: SCTP_PEER_ADDR_CHANGE\n");
			}
			break;
		    case SCTP_SEND_FAILED:
			if ( ctx->debug ) {
			    fprintf(stderr, "  * Notification: SCTP_SEND_FAILED\n");
			}
			break;
		    case SCTP_REMOTE_ERROR:
			if ( ctx->debug ) {
			    fprintf(stderr, "  * Notification: SCTP_REMOTE_ERROR\n");
			}
			break;
		    case SCTP_SHUTDOWN_EVENT:
			if ( ctx->debug ) {
			    fprintf(stderr, "  * Notification: SCTP_SHUTDOWN_EVENT\n");
			}
			break;
		    default:
			if ( ctx->debug ) {
			    fprintf(stderr, "  * Notification: unknown\n");
			}
			break;
		}

	    }

	    // SCTP notification only, do not forward
	    // Note: nfcapd running on FreeBSD can go mad if such a control message is forwarded
	    //       /var/log/messages is filled up with hundred thousands of error messages.
	    if ( sctp_control == 1 ) { continue; }

	    udp_send(ctx, sctx, fpdu_sctp, &remote_address, n, len);

/*	    bzero(&sstat, sizeof (sstat));
	    ulen = sizeof(sstat);
	    if ( sctp_opt_info(ctx->connsock, assoc_id, SCTP_STATUS, &sstat, &ulen) ) {
		fprintf(stderr, "sctp_opt_info(): %s\n", strerror(errno));
	    }
	    fprintf(stderr, "  Stream: %d\n", sndrcvinfo.sinfo_stream);
	    fprintf(stderr, "  Stream Seq Number: %d\n", sndrcvinfo.sinfo_ssn);
	    fprintf(stderr, "  SCTP flags: 0x%X\n", sndrcvinfo.sinfo_flags);
	    fprintf(stderr, "  Payload Protocol ID: %d\n", sndrcvinfo.sinfo_ppid);
	    fprintf(stderr, "  Transmission Seq Number: 0x%X\n", sndrcvinfo.sinfo_tsn);
	    fprintf(stderr, "  Cumulative TSN: %u\n", sndrcvinfo.sinfo_cumtsn);
	    fprintf(stderr, "  Association ID: %u\n", assoc_id);

	    struct sctp_assocparams sctp_assoc;
	    ulen = sizeof(sctp_assoc);
	    assoc_id = 0;
	    if ( sctp_opt_info(ctx->connsock, assoc_id, SCTP_ASSOCINFO, &sctp_assoc, &ulen) ) {
		fprintf(stderr, "sctp_opt_info(): %s\n", strerror(errno));
	    }
*/

	}

    }
}

static void
usage (progname)
     const char *progname;
{
  fprintf (stderr, "Usage: %s [option...] receiver...\n\
\n\
Supported options:\n\
\n\
  -p <port>                UDP/SCTP port to accept flows on (default %d)\n\
  -s <address>             Interface address to accept flows on (default any)\n\
  -d <level>               debug level\n\
  -b <size>                set socket buffer size (default %lu)\n\
  -n			   don't compute UDP checksum (leave at 0)\n\
  -S                       maintain (spoof) source addresses\n\
  -x <delay>               transmit delay in microseconds\n\
  -c configfile            specify a config file to read\n\
  -f                       fork program into background\n\
  -t			   use SCTP protocol on receive side instead of UDP\n\
  -h                       print this usage message and exit\n\
\n\
Specifying receivers:\n\
\n\
  A.B.C.D[%cport[%cfreq][%cttl]]...\n\
where:\n\
  A.B.C.D                  is the receiver's IP address\n\
  port                     is the UDP port to send to (default %d)\n\
  freq                     is the sampling rate (default 1)\n\
  ttl                      is the sending packets TTL value (default %d)\n\
\n\
Config file format:\n\
\n\
  [sctp] a.b.c.d[/e.f.g.h]: receiver ...\n\
where:\n\
  sctp                     use SCTP for this specific source\n\
  a.b.c.d                  is the senders IP address\n\
  e.f.g.h                  is a mask to apply to the sender (default 255.255.255.255)\n\
  receiver                 see above.\n\
\n\
Receivers specified on the command line will get all packets, those\n\
specified in the config-file will get only packets with a matching source.\n\n\
",
	   progname,
	   FLOWPORT, (unsigned long) DEFAULT_SOCKBUFLEN,
	   PORT_SEPARATOR, FREQ_SEPARATOR, TTL_SEPARATOR,
	   FLOWPORT,
	   DEFAULT_TTL);
}

static int
send_pdu_to_peer (peer, fpdu, length, source_addr)
     struct peer * peer;
     const void * fpdu;
     size_t length;
     struct sockaddr_in * source_addr;
{
  if (peer->flags & pf_SPOOF)
    {
      int rawsend_flags
	= ((peer->flags & pf_CHECKSUM) ? RAWSEND_COMPUTE_UDP_CHECKSUM : 0);
      return raw_send_from_to (peer->fd, fpdu, length,
			       source_addr, &peer->addr, peer->ttl,
			       rawsend_flags);
    }
  else
    {
      return sendto (peer->fd, (char*) fpdu, length, 0,
		     (struct sockaddr*) &peer->addr,
		     sizeof (struct sockaddr_in));
    }
}

static int
make_cooked_udp_socket (sockbuflen)
     long sockbuflen;
{
  int s;
  if ((s = socket (PF_INET, SOCK_DGRAM, 0)) == -1)
    return s;
  if (sockbuflen != -1)
    {
      if (setsockopt (s, SOL_SOCKET, SO_SNDBUF,
		      (char *) &sockbuflen, sizeof sockbuflen) == -1)
	{
	  fprintf (stderr, "setsockopt(SO_SNDBUF,%ld): %s\n",
		   sockbuflen, strerror (errno));
	}
    }
  return s;
}
